﻿using System;
using System.Collections.Generic;
using System.Text;
using NUnit.Framework;
using Inet.Viewer.Data;
using Inet.Viewer.test.WinForms;

namespace Inet.Viewer.WinForms
{
    [TestFixture]
    class TestReportViewer
    {    
        private const string validServerUrl               = "http://dell28:9000/?report=repo:%2Ftest%2Ffeature%2Freports%2Fsubreports%2FSubreport3.rpt";
        private const string validServerUrlWithParameters = "http://dell28:9000/?report=repo:%2Fcodereview_overview_old2.rpt&hasgrouptree=false&promptonrefresh=true";
        // base url:                                         http://dell28:9000/?report=repo:%2Ftest%2Ffeature%2Freports%2Fsubreports%2FSubreport3.rpt
        // drill down
        private const string validServerSubreport =         "http://dell28:9000/?report=repo:%2Ftest%2Ffeature%2Freports%2Fsubreports%2FSubreport3.rpt&locale=de-DE&subreport=%26type%3d3%26instance%3d46%26sf%3d%2528%257bOrders.ShipCountry%257d%253d%2527Ireland%2527%2529%26group%3d2&export_fmt=html&page=0.json";
        // subreport
        private const string validServerSubreportOnDemand = "http://dell28:9000/?init=html&report=repo%3a%2ftest%2ffeature%2freports%2fsubreports%2fsubreport3.rpt&locale=de&subreport_ondemand=type%3d0%26subreport%3d0";


        [Test]
        [STAThread]
        public void TestWithViewerEqualNull()
        {
            ReportViewer viewer = new ReportViewer();
            ReportView view = new ReportView();
            viewer.AddReportView(view);
            Assert.AreEqual(viewer.CurrentReportView, view);
        }

        [Test]
        [STAThread]
        public void AddReport()
        {
            URLRenderData data = new URLRenderData(validServerUrl);            

            ReportViewer viewer = new ReportViewer();
            Assert.AreEqual(0, viewer.ReportViewCount);
            IReportView view = viewer.AddNewReportView(data);
            
            Assert.AreEqual(viewer.CurrentReportView, view, "When a ReportView is added it is set to the CurrentView by default");
            Assert.AreEqual(1, viewer.ReportViewCount);

            // Add a report that is null
            viewer.AddReportView(null);
            Assert.AreEqual(viewer.CurrentReportView, view, "Null is not allowed to add, but if done, nothing will happen");
            Assert.AreEqual(1, viewer.ReportViewCount);
            // add the same view twice
            try
            {
                viewer.AddReportView(view);
                Assert.Fail("This report was added already and should throw an Argument Exception");
            }
            catch (ArgumentException)
            {
                Assert.Pass("Has thrown an ArgumentException, which is correct");

            }
        }

        [Test]
        [STAThread]
        public void RemoveReport()
        {
            URLRenderData data = new URLRenderData(validServerUrl);            
            ReportView view = new ReportView();
            view.ReportData = data;

            ReportViewer viewer = new ReportViewer();
            //
            viewer.RemoveReportView(view);
            Assert.IsNull(viewer.CurrentReportView, "Should be null as no view was added yet");

            Assert.AreEqual(0, viewer.ReportViewCount);
            viewer.AddReportView(view);
            Assert.AreEqual(1, viewer.ReportViewCount);
            viewer.CurrentReportView = null;
            viewer.RemoveReportView(view);
            Assert.AreEqual(0, viewer.ReportViewCount);
            // what happens to the CurrentView
            Assert.IsNull(viewer.CurrentReportView, "As there was only one view added it should be null");

            // remove even this view is already removed
            viewer.CurrentReportView = view;
            viewer.RemoveReportView(view);
            Assert.IsNull(viewer.CurrentReportView, "As there is no view added it should be null");
        }

        [Test]
        [STAThread]
        public void RemoveAll()
        {
            URLRenderData data1 = new URLRenderData(validServerUrl);
            URLRenderData data2 = new URLRenderData(validServerUrlWithParameters);
            

            ReportViewer viewer = new ReportViewer();
            //
            ReportView view = new ReportView();
            view.ReportData = data1;
            viewer.AddReportView(view);
            Assert.AreEqual(viewer.CurrentReportView, view);

            ReportView view2 = new ReportView();
            view2.ReportData = data2;
            viewer.AddReportView(view2);
            Assert.AreEqual(viewer.CurrentReportView, view2);

            viewer.CloseAllReportViews();
            Assert.IsNull(viewer.CurrentReportView, "As there is no view added it should be null");
        }

        [Test]
        [STAThread]
        public void TestCurrentView()
        {
            ReportViewer viewer = new ReportViewer();
            Assert.IsNull(viewer.CurrentReportView, "CurrentView should be null as no view was added so far ");

            URLRenderData data = new URLRenderData(validServerUrl);            
            ReportView view = new ReportView();
            view.ReportData = data;

            viewer.CurrentReportView = view;
            Assert.IsNull(viewer.CurrentReportView, "CurrentView should be null as only views that have been added before can be set as CurrentView ");
        }      

        [Test]
        [STAThread]
        public void TestAddReportViews()
        {
            URLRenderData dataSimpleUrl = new URLRenderData(validServerUrl);
            URLRenderData dataUrlWithParameter = new URLRenderData(validServerUrlWithParameters);
            URLRenderData dataUrlWithSubreport = new URLRenderData(validServerSubreport);
            URLRenderData dataUrlWithSubreportOnDemand = new URLRenderData(validServerSubreportOnDemand);

            ReportView viewSimple = new ReportView();
            viewSimple.ReportData = dataSimpleUrl;
            ReportView viewParameters = new ReportView();
            viewParameters.ReportData = dataUrlWithParameter;
            ReportView viewSubreport = new ReportView();
            viewSubreport.ReportData = dataUrlWithSubreport;
            ReportView viewSubreportOnDemand = new ReportView();
            viewSubreportOnDemand.ReportData = dataUrlWithSubreportOnDemand;

            ReportViewer viewer = new ReportViewer();

            viewer.AddReportView(viewSimple);
            Assert.AreEqual(viewSimple, viewer.CurrentReportView, "current view");
            Assert.AreEqual(1, viewer.ReportViewCount);
            viewer.AddReportView(viewParameters);
            Assert.AreEqual(viewParameters, viewer.CurrentReportView, "current view");
            Assert.AreEqual(2, viewer.ReportViewCount);
            viewer.AddReportView(viewSubreport);
            Assert.AreEqual(viewSubreport, viewer.CurrentReportView, "current view");
            Assert.AreEqual(3, viewer.ReportViewCount);          
            viewer.AddReportView(viewSubreportOnDemand);
            Assert.AreEqual(viewSubreportOnDemand, viewer.CurrentReportView, "current view");            
            
        }        
    }
}
